<?php

use App\Models\User;
use App\Services\Facades\Option;
use App\Services\Hook;
use Blessing\Filter;
use Blessing\Rejection;
use Illuminate\Console\Events\ArtisanStarting;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Route;
use Laravel\Passport\Client;
use Laravel\Passport\Passport;
use LittleSkin\YggdrasilConnect\Console\CreatePersonalAccessClient;
use LittleSkin\YggdrasilConnect\Console\FixUUIDTable;
use LittleSkin\YggdrasilConnect\Models\AccessToken;
use LittleSkin\YggdrasilConnect\Models\Profile;
use LittleSkin\YggdrasilConnect\Models\UUID;
use LittleSkin\YggdrasilConnect\Scope;

require __DIR__.'/src/Utils/helpers.php';

return function (Dispatcher $events, Filter $filter, Request $request) {
    Passport::personalAccessTokensExpireIn(now()->addSeconds(intval(Option::get('ygg_token_expire_1'))));

    if (env('YGG_VERBOSE_LOG')) {
        config(['logging.channels.ygg' => [
            'driver' => 'single',
            'path' => storage_path('logs/yggdrasil.log'),
        ]]);
    } else {
        config(['logging.channels.ygg' => [
            'driver' => 'monolog',
            'handler' => Monolog\Handler\NullHandler::class,
        ]]);
    }

    // 记录访问详情
    if (request()->is('api/yggdrasil/*')) {
        ygg_log_http_request_and_response();
    }

    $filter->add('can_add_player', function ($can, string $name) {
        if (option('ygg_uuid_algorithm') === 'v3') {
            $uuid = UUID::generateUuidV3($name);
            if (UUID::where('uuid', $uuid)->orWhere('name', $name)->count()) {
                return new Rejection('名称 已经被占用。');
            }
        } else {
            if (UUID::where('name', $name)->count()) {
                return new Rejection('UUID 表数据错误，请联系站点管理员处理');
            }
        }

        return $can;
    });

    // 向用户中心首页添加「快速配置启动器」板块
    if (option('ygg_show_config_section')) {
        $filter->add('grid:user.index', function ($grid) {
            $grid['widgets'][0][0][] = 'LittleSkin\YggdrasilConnect::dnd';

            return $grid;
        });
        Hook::addScriptFileToPage(plugin('yggdrasil-connect')->assets('dnd.js'), ['user']);
    }

    $events->listen(ArtisanStarting::class, function (ArtisanStarting $event) {
        $event->artisan->resolveCommands([
            CreatePersonalAccessClient::class,
            FixUUIDTable::class,
        ]);
    });
    $events->listen('user.profile.updated', function (User $user, string $action) {
        if ($action === 'password' || $action === 'email') {
            AccessToken::revokeAllForUser($user);
        }
    });
    $events->listen('player.added', 'LittleSkin\\YggdrasilConnect\\Listeners\\OnPlayerAdded@handle');
    $events->listen('player.renamed', 'LittleSkin\\YggdrasilConnect\\Listeners\\OnPlayerRenamed@handle');

    // MODIFICATION: UNION
    $events->listen(App\Events\PlayerWasAdded::class, function ($event) {
        $player = $event->player;
        $uuid = Profile::getUuidFromName($player->name);
        $response = Http::timeout(5.0)->withHeaders([ 'X-Union-Member-Key' => option('union_member_key')])
            ->post(option('union_api_root').'/profile',
                [ 'id' => $uuid, 'name' => $player->name ]
            );
        if (!$response->successful()) {
            Log::channel('ygg')->info("Player [$player->name] sync to union failed.");
        }
        Log::channel('ygg')->info("Player [$player->name] is added.");
    });
    
    $events->listen(App\Events\PlayerProfileUpdated::class, function ($event) {
        $player = $event->player;
        $uuid = Profile::getUuidFromName($player->name);
        $response = Http::timeout(5.0)->withHeaders([ 'X-Union-Member-Key' => option('union_member_key')])
            ->put(option('union_api_root').'/profile/'.$uuid,
                [ 'name' => $player->name ]
            );
        if (!$response->successful()) {
            Log::channel('ygg')->info("Player [$player->name] sync to union failed.");
        }
        Log::channel('ygg')->info("Player [$uuid]'s name is changed to [$player->name]. ");
    });

    $events->listen(App\Events\PlayerWillBeDeleted::class, function ($event) {
        $player = $event->player;
        $uuid = Profile::getUuidFromName($player->name);
        $response = Http::timeout(5.0)->withHeaders([ 'X-Union-Member-Key' => option('union_member_key')])
            ->delete(option('union_api_root').'/profile/'.$uuid);
        if (!$response->successful()) {
            Log::channel('ygg')->info("Player [$player->name] sync to union failed.");
        }
        Log::channel('ygg')->info("Player [$player->name] is deleted.");
    });

    if (env('YGG_VERBOSE_LOG')) {
        Hook::addMenuItem('admin', 4, [
            'title' => 'LittleSkin\\YggdrasilConnect::log.title',
            'link' => 'admin/yggdrasil-log',
            'icon' => 'fa-history',
        ]);
    }

    Hook::addRoute(function () {
        Route::namespace('LittleSkin\YggdrasilConnect\Controllers')
            ->group(function () {
                Route::prefix('api/yggdrasil')->group(__DIR__.'/routes.php');

                Route::prefix('admin')->middleware(['web', 'auth'])->group(function () {
                    Route::middleware('role:admin')
                        ->group(function () {
                            Route::get('yggdrasil-log', 'ConfigController@logPage');
                            Route::post(
                                'plugins/config/yggdrasil-connect/generate',
                                'ConfigController@generate'
                            );
                        });
                });
                Route::prefix('yggc')->group(function () {
                    Route::middleware(['web', 'LittleSkin\YggdrasilConnect\Middleware\CheckIsIssuerSet', 'auth'])->group(function () {
                        Route::get('callback', 'OIDCController@passportCallback');
                        Route::post('callback', 'OIDCController@selectProfile');
                        Route::get('cancel', 'OIDCController@cancel');
                    });
                    // Route::get('device', 'OIDCController@userCode');
                    Route::middleware(['api', 'LittleSkin\YggdrasilConnect\Middleware\CheckBearerTokenOAuth:'.Scope::OPENID])
                        ->get('userinfo', 'OIDCController@getUserInfo');
                });
                
                // MODIFICATION: UNION
                Route::middleware(['LittleSkin\YggdrasilConnect\Middleware\UnionHostVerify'])
                    ->prefix('api/union/member')
                    ->group(function () {
                        Route::post('updatelist', 'UnionController@updateList');
                        Route::post('updateprivatekey', 'UnionController@updatePrivateKey');
                        Route::post('updatebackendkey', 'UnionController@serverUpdatesBackendKey');
                        Route::post('sync', 'UnionController@triggerSync');
                        Route::post('remapuuid', 'UnionController@remapUUID');
                        Route::post('updateplugin', 'UpdateController@update');
                    
                        Route::get('queryemail','UnionBlacklistController@queryEmail');
                        Route::post('diagnose', 'UnionController@diagnose');
                    });

                Route::prefix('api/union/member')
                    ->group(function () {
                        Route::get('/', 'UnionController@hello');
                    });

                Route::prefix('api/union/member/oauth2')
                    ->group(function () {
                        Route::get('/', 'UnionOAuth2Controller@getSigPublicKey');
                        Route::get('grant', 'UnionOAuth2Controller@grant');
                    });
        
                Route::middleware(['web', 'auth'])
                    ->prefix('union')
                    ->group(function () {
                        Route::get('/', 'UnionProfileController@render');
                        Route::post('bind', 'UnionProfileController@bind');
                        Route::post('bindto', 'UnionProfileController@bindto');
                        Route::post('unbind', 'UnionProfileController@unbind');
                        Route::post('remapuuid', 'UnionProfileController@requestRemapUUID');
                        Route::get('security/level', 'UnionController@getSecurityLevel');
                    });

                Route::middleware(['web', 'auth', 'role:admin'])
                    ->prefix('admin/union')
                    ->group(function () {
                        Route::post('member/updatelist', 'UnionController@updateList');
                        Route::post('member/updateprivatekey', 'UnionController@updatePrivateKey');
                        Route::post('member/sync', 'UnionController@triggerSync');
                        Route::post('member/diagnose', 'UnionController@triggerDiagnose');
                    
                        Route::view('view/blacklist', 'LittleSkin\\YggdrasilConnect::blacklist');
                        Route::get('view/blacklist/list', 'UnionBlacklistController@viewBlacklist');
                        Route::post('blacklist/create', 'UnionBlacklistController@create');
                        Route::post('blacklist/invalidate/{id}', 'UnionBlacklistController@invalidate');
                        Route::post('blacklist/delete/{id}', 'UnionBlacklistController@delete');
                    
                    });
            });
    });

    Hook::addMenuItem('explore', 1, [
        'title' => 'LittleSkin\\YggdrasilConnect::union.title',
        'link' => 'union',
        'icon' => 'fa-paper-plane',
    ]);
  
  	Hook::addMenuItem('admin', 14, [
      'title' => 'LittleSkin\\YggdrasilConnect::union.blacklist.title',
      'link'  => '/admin/union/view/blacklist',
      'icon'  => 'fa-ban',
    ]);

    Hook::pushMiddleware('LittleSkin\\YggdrasilConnect\\Middleware\\HandleCors'); // https://t.me/blessing_skin/184887

    if (option('ygg_enable_ali')) {
        Hook::pushMiddleware('LittleSkin\YggdrasilConnect\Middleware\AddApiIndicationHeader');
    }

    if ($request->is('oauth/authorize')) {
        Hook::pushMiddleware('LittleSkin\YggdrasilConnect\Middleware\CheckIfScopeValid');
    }

    Client::retrieved(function (Client $client) use ($request) {
        if ($request->is('oauth/authorize')) {
            $yggc_redirect = option('site_url').'/yggc/callback';
            if (!in_array($yggc_redirect, explode(',', $client->redirect))) {
                $client->redirect = "$client->redirect,$yggc_redirect";
            }
        }
    });
};
